<?php

namespace App\Http\Controllers;

use App\BusinessLocation;
use App\Charts\CommonChart;
use App\Currency;
use App\Media;
use App\Transaction;
use App\User;
use App\Utils\BusinessUtil;
use App\Utils\ModuleUtil;
use App\Utils\RestaurantUtil;
use App\Utils\TransactionUtil;
use App\Utils\ProductUtil;
use App\Utils\Util;
use App\VariationLocationDetails;
use Datatables;
use DB;
use Illuminate\Http\Request;
use Illuminate\Notifications\DatabaseNotification;
use Carbon\Carbon;

class HomeController extends Controller
{
    /**
     * All Utils instance.
     */
    protected $businessUtil;
    protected $transactionUtil;
    protected $moduleUtil;
    protected $commonUtil;
    protected $restUtil;
    protected $productUtil;

    public function __construct(
        BusinessUtil $businessUtil,
        TransactionUtil $transactionUtil,
        ModuleUtil $moduleUtil,
        Util $commonUtil,
        RestaurantUtil $restUtil,
        ProductUtil $productUtil,
    ) {
        $this->businessUtil = $businessUtil;
        $this->transactionUtil = $transactionUtil;
        $this->moduleUtil = $moduleUtil;
        $this->commonUtil = $commonUtil;
        $this->restUtil = $restUtil;
        $this->productUtil = $productUtil;
    }

    /**
     * Show the application dashboard.
     */
    public function index()
    {
        $user = auth()->user();
        if ($user->user_type == 'user_customer') {
            return redirect()->action([\Modules\Crm\Http\Controllers\DashboardController::class, 'index']);
        }

        $business_id = request()->session()->get('user.business_id');
        $is_admin = $this->businessUtil->is_admin(auth()->user());

        if (! auth()->user()->can('dashboard.data')) {
            return view('home.index');
        }

        $fy = $this->businessUtil->getCurrentFinancialYear($business_id);
        $currency = Currency::where('id', request()->session()->get('business.currency_id'))->first();

        // Chart logic (your existing code)
        $least_30_days = \Carbon::parse($fy['start'])->subDays(30)->format('Y-m-d');
        $sells_this_fy = $this->transactionUtil->getSellsCurrentFy($business_id, $least_30_days, $fy['end']);
        $all_locations = BusinessLocation::forDropdown($business_id)->toArray();

        // Charts creation (your existing chart code)
        $labels = [];
        $all_sell_values = [];
        $dates = [];
        for ($i = 29; $i >= 0; $i--) {
            $date = \Carbon::now()->subDays($i)->format('Y-m-d');
            $dates[] = $date;
            $labels[] = date('j M Y', strtotime($date));
            $total_sell_on_date = $sells_this_fy->where('date', $date)->sum('total_sells');
            $all_sell_values[] = !empty($total_sell_on_date) ? (float) $total_sell_on_date : 0;
        }

        // Location sells grouping (your existing code)
        $location_sells = [];
        foreach ($all_locations as $loc_id => $loc_name) {
            $values = [];
            foreach ($dates as $date) {
                $total_sell_on_date_location = $sells_this_fy->where('date', $date)->where('location_id', $loc_id)->sum('total_sells');
                $values[] = !empty($total_sell_on_date_location) ? (float) $total_sell_on_date_location : 0;
            }
            $location_sells[$loc_id]['loc_label'] = $loc_name;
            $location_sells[$loc_id]['values'] = $values;
        }

        $sells_chart_1 = new CommonChart;
        $sells_chart_1->labels($labels)
            ->options($this->__chartOptions(__('home.total_sells', ['currency' => $currency->code])));

        if (!empty($location_sells)) {
            foreach ($location_sells as $location_sell) {
                $sells_chart_1->dataset($location_sell['loc_label'], 'line', $location_sell['values']);
            }
        }

        if (count($all_locations) > 1) {
            $sells_chart_1->dataset(__('report.all_locations'), 'line', $all_sell_values);
        }

        // FY chart (your existing code for sells_chart_2)
        $labels = [];
        $values = [];
        $date = strtotime($fy['start']);
        $last = date('m-Y', strtotime($fy['end']));
        $fy_months = [];
        do {
            $month_year = date('m-Y', $date);
            $fy_months[] = $month_year;
            $labels[] = \Carbon::createFromFormat('m-Y', $month_year)->format('M-Y');
            $date = strtotime('+1 month', $date);
            $total_sell_in_month_year = $sells_this_fy->where('yearmonth', $month_year)->sum('total_sells');
            $values[] = !empty($total_sell_in_month_year) ? (float) $total_sell_in_month_year : 0;
        } while ($month_year != $last);

        $fy_sells_by_location_data = [];
        foreach ($all_locations as $loc_id => $loc_name) {
            $values_data = [];
            foreach ($fy_months as $month) {
                $total_sell_in_month_year_location = $sells_this_fy->where('yearmonth', $month)->where('location_id', $loc_id)->sum('total_sells');
                $values_data[] = !empty($total_sell_in_month_year_location) ? (float) $total_sell_in_month_year_location : 0;
            }
            $fy_sells_by_location_data[$loc_id]['loc_label'] = $loc_name;
            $fy_sells_by_location_data[$loc_id]['values'] = $values_data;
        }

        $sells_chart_2 = new CommonChart;
        $sells_chart_2->labels($labels)
            ->options($this->__chartOptions(__('home.total_sells', ['currency' => $currency->code])));
        if (!empty($fy_sells_by_location_data)) {
            foreach ($fy_sells_by_location_data as $location_sell) {
                $sells_chart_2->dataset($location_sell['loc_label'], 'line', $location_sell['values']);
            }
        }
        if (count($all_locations) > 1) {
            $sells_chart_2->dataset(__('report.all_locations'), 'line', $values);
        }

        // Get initial dashboard data for today
        $today_data = $this->getDashboardTotals();

        $module_widgets = $this->moduleUtil->getModuleData('dashboard_widget');
        $widgets = [];
        foreach ($module_widgets as $widget_array) {
            if (!empty($widget_array['position'])) {
                $widgets[$widget_array['position']][] = $widget_array['widget'];
            }
        }

        $common_settings = !empty(session('business.common_settings')) ? session('business.common_settings') : [];

        return view('home.index', compact(
            'sells_chart_1',
            'sells_chart_2',
            'widgets',
            'all_locations',
            'common_settings',
            'is_admin',
            'today_data'
        ));
    }

    /**
     * Get dashboard totals with comparison data
     */
    public function getDashboardTotals($start = null, $end = null, $location_id = null)
    {
        $business_id = request()->session()->get('user.business_id');
        $created_by = request()->user_id;

        // Default to today if no dates provided
        if (!$start || !$end) {
            $start = \Carbon::today()->format('Y-m-d');
            $end = \Carbon::today()->format('Y-m-d');
        }

        // Calculate comparison period (same length, immediately before)
        $start_carbon = \Carbon::parse($start);
        $end_carbon = \Carbon::parse($end);
        $period_length = $start_carbon->diffInDays($end_carbon);

        $prev_end = $start_carbon->copy()->subDay();
        $prev_start = $prev_end->copy()->subDays($period_length);

        // Get current period data
        $current_purchase = $this->transactionUtil->getPurchaseTotals($business_id, $start, $end, $location_id, $created_by);
        $current_sell = $this->transactionUtil->getSellTotals($business_id, $start, $end, $location_id, $created_by);
        $current_ledger_discount = $this->transactionUtil->getTotalLedgerDiscount($business_id, $start, $end);

        // Get previous period data for comparison
        $prev_purchase = $this->transactionUtil->getPurchaseTotals($business_id, $prev_start->format('Y-m-d'), $prev_end->format('Y-m-d'), $location_id, $created_by);
        $prev_sell = $this->transactionUtil->getSellTotals($business_id, $prev_start->format('Y-m-d'), $prev_end->format('Y-m-d'), $location_id, $created_by);

        // Get transaction totals
        $transaction_types = ['purchase_return', 'sell_return', 'expense'];
        $current_transactions = $this->transactionUtil->getTransactionTotals($business_id, $transaction_types, $start, $end, $location_id, $created_by);
        $prev_transactions = $this->transactionUtil->getTransactionTotals($business_id, $transaction_types, $prev_start->format('Y-m-d'), $prev_end->format('Y-m-d'), $location_id, $created_by);

        // Calculate current totals
        $total_sell = !empty($current_sell['total_sell_inc_tax']) ? $current_sell['total_sell_inc_tax'] : 0;
        $total_expense = !empty($current_transactions['total_expense']) ? $current_transactions['total_expense'] : 0;
        $invoice_due = ($current_sell['invoice_due'] ?? 0) - $current_ledger_discount['total_sell_discount'];
        $total_sell_return = !empty($current_transactions['total_sell_return_inc_tax']) ? $current_transactions['total_sell_return_inc_tax'] : 0;
        $total_purchase = !empty($current_purchase['total_purchase_inc_tax']) ? $current_purchase['total_purchase_inc_tax'] : 0;
        $purchase_due = ($current_purchase['purchase_due'] ?? 0) - $current_ledger_discount['total_purchase_discount'];
        $total_purchase_return = !empty($current_transactions['total_purchase_return_inc_tax']) ? $current_transactions['total_purchase_return_inc_tax'] : 0;

        // Calculate net profit
        $net = $total_sell - $invoice_due - $total_expense;

        // Calculate previous totals for comparison
        $prev_total_sell = !empty($prev_sell['total_sell_inc_tax']) ? $prev_sell['total_sell_inc_tax'] : 0;
        $prev_total_expense = !empty($prev_transactions['total_expense']) ? $prev_transactions['total_expense'] : 0;
        $prev_invoice_due = $prev_sell['invoice_due'] ?? 0;
        $prev_net = $prev_total_sell - $prev_invoice_due - $prev_total_expense;
        $prev_total_sell_return = !empty($prev_transactions['total_sell_return_inc_tax']) ? $prev_transactions['total_sell_return_inc_tax'] : 0;
        $prev_total_purchase = !empty($prev_purchase['total_purchase_inc_tax']) ? $prev_purchase['total_purchase_inc_tax'] : 0;
        $prev_purchase_due = $prev_purchase['purchase_due'] ?? 0;
        $prev_total_purchase_return = !empty($prev_transactions['total_purchase_return_inc_tax']) ? $prev_transactions['total_purchase_return_inc_tax'] : 0;

        // Calculate percentage changes
        $sell_change = $this->calculatePercentageChange($total_sell, $prev_total_sell);
        $net_change = $this->calculatePercentageChange($net, $prev_net);
        $invoice_due_change = $this->calculatePercentageChange($invoice_due, $prev_invoice_due);
        $sell_return_change = $this->calculatePercentageChange($total_sell_return, $prev_total_sell_return);
        $purchase_change = $this->calculatePercentageChange($total_purchase, $prev_total_purchase);
        $purchase_due_change = $this->calculatePercentageChange($purchase_due, $prev_purchase_due);
        $purchase_return_change = $this->calculatePercentageChange($total_purchase_return, $prev_total_purchase_return);
        $expense_change = $this->calculatePercentageChange($total_expense, $prev_total_expense);

        return [
            // Current values
            'total_sell' => $total_sell,
            'net' => $net,
            'invoice_due' => $invoice_due,
            'total_sell_return' => $total_sell_return,
            'total_purchase' => $total_purchase,
            'purchase_due' => $purchase_due,
            'total_purchase_return' => $total_purchase_return,
            'total_expense' => $total_expense,

            // Additional data
            'total_sell_return_paid' => $this->transactionUtil->getTotalSellReturnPaid($business_id, $start, $end, $location_id),
            'total_purchase_return_paid' => $this->transactionUtil->getTotalPurchaseReturnPaid($business_id, $start, $end, $location_id),

            // Percentage changes
            'sell_change' => $sell_change,
            'net_change' => $net_change,
            'invoice_due_change' => $invoice_due_change,
            'sell_return_change' => $sell_return_change,
            'purchase_change' => $purchase_change,
            'purchase_due_change' => $purchase_due_change,
            'purchase_return_change' => $purchase_return_change,
            'expense_change' => $expense_change,
        ];
    }

    /**
     * Calculate percentage change between two values
     */
    private function calculatePercentageChange($current, $previous)
    {
        if ($previous == 0) {
            return $current > 0 ? 100 : 0;
        }

        $change = (($current - $previous) / abs($previous)) * 100;
        return round($change, 1);
    }

    /**
     * Updated getTotals method to work with AJAX
     */
    public function getTotals()
    {
        if (request()->ajax()) {
            $start = request()->start;
            $end = request()->end;
            $location_id = request()->location_id;

            return $this->getDashboardTotals($start, $end, $location_id);
        }
    }

    /**
     * Retrieves sell products whose available quntity is less than alert quntity.
     *
     * @return \Illuminate\Http\Response
     */
    public function getProductStockAlert()
    {
        if (request()->ajax()) {
            $business_id = request()->session()->get('user.business_id');
            $permitted_locations = auth()->user()->permitted_locations();
            $products = $this->productUtil->getProductAlert($business_id, $permitted_locations);

            return Datatables::of($products)
                ->editColumn('product', function ($row) {
                    if ($row->type == 'single') {
                        return $row->product . ' (' . $row->sku . ')';
                    } else {
                        return $row->product . ' - ' . $row->product_variation . ' - ' . $row->variation . ' (' . $row->sub_sku . ')';
                    }
                })
                ->editColumn('stock', function ($row) {
                    $stock = $row->stock ? $row->stock : 0;

                    return '<span data-is_quantity="true" class="display_currency" data-currency_symbol=false>' . (float) $stock . '</span> ' . $row->unit;
                })
                ->removeColumn('sku')
                ->removeColumn('sub_sku')
                ->removeColumn('unit')
                ->removeColumn('type')
                ->removeColumn('product_variation')
                ->removeColumn('variation')
                ->rawColumns([2])
                ->make(false);
        }
    }

    /**
     * Retrieves payment dues for the purchases.
     *
     * @return \Illuminate\Http\Response
     */
    public function getPurchasePaymentDues()
    {
        if (request()->ajax()) {
            $business_id = request()->session()->get('user.business_id');
            $today = \Carbon::now()->format('Y-m-d H:i:s');

            $query = Transaction::join(
                'contacts as c',
                'transactions.contact_id',
                '=',
                'c.id'
            )
                ->leftJoin(
                    'transaction_payments as tp',
                    'transactions.id',
                    '=',
                    'tp.transaction_id'
                )
                ->where('transactions.business_id', $business_id)
                ->where('transactions.type', 'purchase')
                ->where('transactions.payment_status', '!=', 'paid')
                ->whereRaw("DATEDIFF( DATE_ADD( transaction_date, INTERVAL IF(transactions.pay_term_type = 'days', transactions.pay_term_number, 30 * transactions.pay_term_number) DAY), '$today') <= 7");

            //Check for permitted locations of a user
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $query->whereIn('transactions.location_id', $permitted_locations);
            }

            if (! empty(request()->input('location_id'))) {
                $query->where('transactions.location_id', request()->input('location_id'));
            }

            $dues = $query->select(
                'transactions.id as id',
                'c.name as supplier',
                'c.supplier_business_name',
                'ref_no',
                'final_total',
                DB::raw('SUM(tp.amount) as total_paid')
            )
                ->groupBy('transactions.id');

            return Datatables::of($dues)
                ->addColumn('due', function ($row) {
                    $total_paid = ! empty($row->total_paid) ? $row->total_paid : 0;
                    $due = $row->final_total - $total_paid;

                    return '<span class="display_currency" data-currency_symbol="true">' .
                        $due . '</span>';
                })
                ->addColumn('action', '@can("purchase.create") <a href="{{action([\App\Http\Controllers\TransactionPaymentController::class, \'addPayment\'], [$id])}}" class="tw-dw-btn tw-dw-btn-xs tw-dw-btn-outline tw-dw-btn-accent add_payment_modal"><i class="fas fa-money-bill-alt"></i> @lang("purchase.add_payment")</a> @endcan')
                ->removeColumn('supplier_business_name')
                ->editColumn('supplier', '@if(!empty($supplier_business_name)) {{$supplier_business_name}}, <br> @endif {{$supplier}}')
                ->editColumn('ref_no', function ($row) {
                    if (auth()->user()->can('purchase.view')) {
                        return  '<a href="#" data-href="' . action([\App\Http\Controllers\PurchaseController::class, 'show'], [$row->id]) . '"
                                    class="btn-modal" data-container=".view_modal">' . $row->ref_no . '</a>';
                    }

                    return $row->ref_no;
                })
                ->removeColumn('id')
                ->removeColumn('final_total')
                ->removeColumn('total_paid')
                ->rawColumns([0, 1, 2, 3])
                ->make(false);
        }
    }

    /**
     * Retrieves payment dues for the purchases.
     *
     * @return \Illuminate\Http\Response
     */
    public function getSalesPaymentDues()
    {
        if (request()->ajax()) {
            $business_id = request()->session()->get('user.business_id');
            $today = \Carbon::now()->format('Y-m-d H:i:s');

            $query = Transaction::join(
                'contacts as c',
                'transactions.contact_id',
                '=',
                'c.id'
            )
                ->leftJoin(
                    'transaction_payments as tp',
                    'transactions.id',
                    '=',
                    'tp.transaction_id'
                )
                ->where('transactions.business_id', $business_id)
                ->where('transactions.type', 'sell')
                ->where('transactions.payment_status', '!=', 'paid')
                ->whereNotNull('transactions.pay_term_number')
                ->whereNotNull('transactions.pay_term_type')
                ->whereRaw("DATEDIFF( DATE_ADD( transaction_date, INTERVAL IF(transactions.pay_term_type = 'days', transactions.pay_term_number, 30 * transactions.pay_term_number) DAY), '$today') <= 7");

            //Check for permitted locations of a user
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $query->whereIn('transactions.location_id', $permitted_locations);
            }

            if (! empty(request()->input('location_id'))) {
                $query->where('transactions.location_id', request()->input('location_id'));
            }

            $dues = $query->select(
                'transactions.id as id',
                'c.name as customer',
                'c.supplier_business_name',
                'transactions.invoice_no',
                'final_total',
                DB::raw('SUM(tp.amount) as total_paid')
            )
                ->groupBy('transactions.id');

            return Datatables::of($dues)
                ->addColumn('due', function ($row) {
                    $total_paid = ! empty($row->total_paid) ? $row->total_paid : 0;
                    $due = $row->final_total - $total_paid;

                    return '<span class="display_currency" data-currency_symbol="true">' .
                        $due . '</span>';
                })
                ->editColumn('invoice_no', function ($row) {
                    if (auth()->user()->can('sell.view')) {
                        return  '<a href="#" data-href="' . action([\App\Http\Controllers\SellController::class, 'show'], [$row->id]) . '"
                                    class="btn-modal" data-container=".view_modal">' . $row->invoice_no . '</a>';
                    }

                    return $row->invoice_no;
                })
                ->addColumn('action', '@if(auth()->user()->can("sell.create") || auth()->user()->can("direct_sell.access")) <a href="{{action([\App\Http\Controllers\TransactionPaymentController::class, \'addPayment\'], [$id])}}" class="tw-dw-btn tw-dw-btn-xs tw-dw-btn-outline tw-dw-btn-accent add_payment_modal"><i class="fas fa-money-bill-alt"></i> @lang("purchase.add_payment")</a> @endif')
                ->editColumn('customer', '@if(!empty($supplier_business_name)) {{$supplier_business_name}}, <br> @endif {{$customer}}')
                ->removeColumn('supplier_business_name')
                ->removeColumn('id')
                ->removeColumn('final_total')
                ->removeColumn('total_paid')
                ->rawColumns([0, 1, 2, 3])
                ->make(false);
        }
    }

    public function loadMoreNotifications()
    {
        $notifications = auth()->user()->notifications()->orderBy('created_at', 'DESC')->paginate(10);

        if (request()->input('page') == 1) {
            auth()->user()->unreadNotifications->markAsRead();
        }
        $notifications_data = $this->commonUtil->parseNotifications($notifications);

        return view('layouts.partials.notification_list', compact('notifications_data'));
    }

    /**
     * Function to count total number of unread notifications
     *
     * @return json
     */
    public function getTotalUnreadNotifications()
    {
        $unread_notifications = auth()->user()->unreadNotifications;
        $total_unread = $unread_notifications->count();

        $notification_html = '';
        $modal_notifications = [];
        foreach ($unread_notifications as $unread_notification) {
            if (isset($data['show_popup'])) {
                $modal_notifications[] = $unread_notification;
                $unread_notification->markAsRead();
            }
        }
        if (! empty($modal_notifications)) {
            $notification_html = view('home.notification_modal')->with(['notifications' => $modal_notifications])->render();
        }

        return [
            'total_unread' => $total_unread,
            'notification_html' => $notification_html,
        ];
    }

    private function __chartOptions($title)
    {
        return [
            'yAxis' => [
                'title' => [
                    'text' => $title,
                ],
            ],
            'legend' => [
                'align' => 'right',
                'verticalAlign' => 'top',
                'floating' => true,
                'layout' => 'vertical',
                'padding' => 20,
            ],
        ];
    }

    public function getCalendar()
    {
        $business_id = request()->session()->get('user.business_id');
        $is_admin = $this->restUtil->is_admin(auth()->user(), $business_id);
        $is_superadmin = auth()->user()->can('superadmin');
        if (request()->ajax()) {
            $data = [
                'start_date' => request()->start,
                'end_date' => request()->end,
                'user_id' => ($is_admin || $is_superadmin) && ! empty(request()->user_id) ? request()->user_id : auth()->user()->id,
                'location_id' => ! empty(request()->location_id) ? request()->location_id : null,
                'business_id' => $business_id,
                'events' => request()->events ?? [],
                'color' => '#007FFF',
            ];
            $events = [];

            if (in_array('bookings', $data['events'])) {
                $events = $this->restUtil->getBookingsForCalendar($data);
            }

            $module_events = $this->moduleUtil->getModuleData('calendarEvents', $data);

            foreach ($module_events as $module_event) {
                $events = array_merge($events, $module_event);
            }

            return $events;
        }

        $all_locations = BusinessLocation::forDropdown($business_id)->toArray();
        $users = [];
        if ($is_admin) {
            $users = User::forDropdown($business_id, false);
        }

        $event_types = [
            'bookings' => [
                'label' => __('restaurant.bookings'),
                'color' => '#007FFF',
            ],
        ];
        $module_event_types = $this->moduleUtil->getModuleData('eventTypes');
        foreach ($module_event_types as $module_event_type) {
            $event_types = array_merge($event_types, $module_event_type);
        }

        return view('home.calendar')->with(compact('all_locations', 'users', 'event_types'));
    }

    public function showNotification($id)
    {
        $notification = DatabaseNotification::find($id);

        $data = $notification->data;

        $notification->markAsRead();

        return view('home.notification_modal')->with([
            'notifications' => [$notification],
        ]);
    }

    public function attachMediasToGivenModel(Request $request)
    {
        if ($request->ajax()) {
            try {
                $business_id = request()->session()->get('user.business_id');

                $model_id = $request->input('model_id');
                $model = $request->input('model_type');
                $model_media_type = $request->input('model_media_type');

                DB::beginTransaction();

                //find model to which medias are to be attached
                $model_to_be_attached = $model::where('business_id', $business_id)
                    ->findOrFail($model_id);

                Media::uploadMedia($business_id, $model_to_be_attached, $request, 'file', false, $model_media_type);

                DB::commit();

                $output = [
                    'success' => true,
                    'msg' => __('lang_v1.success'),
                ];
            } catch (Exception $e) {
                DB::rollBack();

                \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

                $output = [
                    'success' => false,
                    'msg' => __('messages.something_went_wrong'),
                ];
            }

            return $output;
        }
    }

    public function getUserLocation($latlng)
    {
        $latlng_array = explode(',', $latlng);

        $response = $this->moduleUtil->getLocationFromCoordinates($latlng_array[0], $latlng_array[1]);

        return ['address' => $response];
    }
}
