<?php
/**
 * Add this method to app/Http/Controllers/SellPosController.php
 * Place it before the closing brace of the class
 */

    /**
     * Get product history for POS modal
     * Includes sales, purchases, stock movements, warranty, and expiry info
     *
     * @param  Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getProductHistory(Request $request)
    {
        if (!auth()->user()->can('product.view')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = $request->session()->get('user.business_id');
            $search = $request->input('search');
            $per_page = 8;

            // Check permissions for each section
            $can_view_sales = auth()->user()->can('sell.view') || auth()->user()->can('direct_sell.view');
            $can_view_purchases = auth()->user()->can('purchase.view');
            $can_view_stock = auth()->user()->can('product.view');

            // Pagination parameters for each tab
            $sales_page = $request->input('sales_page', 1);
            $purchases_page = $request->input('purchases_page', 1);
            $stock_page = $request->input('stock_page', 1);
            $warranty_page = $request->input('warranty_page', 1);
            $expiry_page = $request->input('expiry_page', 1);

            if (empty($search)) {
                return response()->json([
                    'success' => false,
                    'msg' => __('lang_v1.please_enter_sku_or_product_name')
                ]);
            }

            // Find the product by SKU or name
            $product = Product::where('business_id', $business_id)
                ->where(function ($query) use ($search) {
                    $query->where('sku', $search)
                        ->orWhere('name', 'like', '%' . $search . '%');
                })
                ->with(['variations', 'warranty', 'unit'])
                ->first();

            // Also search in variations for SKU
            if (!$product) {
                $variation = Variation::whereHas('product', function ($query) use ($business_id) {
                    $query->where('business_id', $business_id);
                })
                    ->where('sub_sku', $search)
                    ->with(['product.warranty', 'product.unit'])
                    ->first();

                if ($variation) {
                    $product = $variation->product;
                }
            }

            if (!$product) {
                return response()->json([
                    'success' => false,
                    'msg' => __('lang_v1.product_not_found')
                ]);
            }

            $variation_ids = $product->variations->pluck('id')->toArray();

            // Get current stock
            $current_stock = DB::table('variation_location_details')
                ->whereIn('variation_id', $variation_ids)
                ->sum('qty_available');

            // Get Sales History with pagination (only if user has permission)
            $sales = collect([]);
            $sales_total = 0;
            if ($can_view_sales) {
                $sales_query = TransactionSellLine::whereIn('variation_id', $variation_ids)
                    ->join('transactions', 'transaction_sell_lines.transaction_id', '=', 'transactions.id')
                    ->join('contacts', 'transactions.contact_id', '=', 'contacts.id')
                    ->join('business_locations', 'transactions.location_id', '=', 'business_locations.id')
                    ->where('transactions.business_id', $business_id)
                    ->whereIn('transactions.type', ['sell'])
                    ->whereIn('transactions.status', ['final'])
                    ->select(
                        'transactions.invoice_no',
                        'transactions.transaction_date',
                        'contacts.name as customer_name',
                        'transaction_sell_lines.quantity',
                        'transaction_sell_lines.unit_price_inc_tax',
                        'business_locations.name as location_name',
                        'transaction_sell_lines.line_discount_amount'
                    )
                    ->orderBy('transactions.transaction_date', 'desc');

                $sales_total = $sales_query->count();
                $sales = $sales_query->skip(($sales_page - 1) * $per_page)->take($per_page)->get();
            }

            // Get Purchase History with pagination (only if user has permission)
            $purchases = collect([]);
            $purchases_total = 0;
            if ($can_view_purchases) {
                $purchases_query = DB::table('purchase_lines')
                    ->whereIn('variation_id', $variation_ids)
                    ->join('transactions', 'purchase_lines.transaction_id', '=', 'transactions.id')
                    ->join('contacts', 'transactions.contact_id', '=', 'contacts.id')
                    ->join('business_locations', 'transactions.location_id', '=', 'business_locations.id')
                    ->where('transactions.business_id', $business_id)
                    ->whereIn('transactions.type', ['purchase'])
                    ->select(
                        'transactions.ref_no',
                        'transactions.transaction_date',
                        'contacts.name as supplier_name',
                        'purchase_lines.quantity',
                        'purchase_lines.purchase_price_inc_tax',
                        'business_locations.name as location_name'
                    )
                    ->orderBy('transactions.transaction_date', 'desc');

                $purchases_total = $purchases_query->count();
                $purchases = $purchases_query->skip(($purchases_page - 1) * $per_page)->take($per_page)->get();
            }

            // Get Stock History (movements) with pagination (only if user has permission)
            $stock_history = [];
            $stock_total = 0;
            if ($can_view_stock) {
                $stock_history_all = $this->productUtil->getVariationStockHistory($business_id, $variation_ids[0] ?? null, null);
                $stock_total = is_array($stock_history_all) ? count($stock_history_all) : 0;
                $stock_history = is_array($stock_history_all)
                    ? array_slice($stock_history_all, ($stock_page - 1) * $per_page, $per_page)
                    : [];
            }

            // Get Warranty Sales (items with warranty) with pagination (only if user has sales permission)
            $warranty_sales = [];
            $warranty_total = 0;
            if ($can_view_sales && $product->warranty_id) {
                $warranty = Warranty::find($product->warranty_id);
                $warranty_query = TransactionSellLine::whereIn('variation_id', $variation_ids)
                    ->join('transactions', 'transaction_sell_lines.transaction_id', '=', 'transactions.id')
                    ->join('contacts', 'transactions.contact_id', '=', 'contacts.id')
                    ->where('transactions.business_id', $business_id)
                    ->whereIn('transactions.type', ['sell'])
                    ->whereIn('transactions.status', ['final'])
                    ->select(
                        'transactions.invoice_no',
                        'transactions.transaction_date',
                        'contacts.name as customer_name',
                        'transaction_sell_lines.quantity'
                    )
                    ->orderBy('transactions.transaction_date', 'desc');

                $warranty_total = $warranty_query->count();
                $warranty_sales = $warranty_query->skip(($warranty_page - 1) * $per_page)->take($per_page)->get()
                    ->map(function ($sale) use ($warranty) {
                        $sale_date = \Carbon::parse($sale->transaction_date);
                        $warranty_end = $warranty ? $warranty->getEndDate($sale->transaction_date) : null;
                        $is_active = $warranty_end ? \Carbon::now()->lt(\Carbon::parse($warranty_end)) : false;

                        return [
                            'invoice_no' => $sale->invoice_no,
                            'transaction_date' => $sale->transaction_date,
                            'customer_name' => $sale->customer_name,
                            'warranty_name' => $warranty ? $warranty->display_name : 'N/A',
                            'warranty_start' => $sale->transaction_date,
                            'warranty_end' => $warranty_end,
                            'is_active' => $is_active
                        ];
                    });
            }

            // Get Expiry Info with pagination (only if user has purchase permission)
            $expiry_info = collect([]);
            $expiry_total = 0;
            if ($can_view_purchases) {
                $expiry_query = DB::table('purchase_lines')
                    ->whereIn('variation_id', $variation_ids)
                    ->join('transactions', 'purchase_lines.transaction_id', '=', 'transactions.id')
                    ->join('business_locations', 'transactions.location_id', '=', 'business_locations.id')
                    ->where('transactions.business_id', $business_id)
                    ->whereNotNull('purchase_lines.exp_date')
                    ->select(
                        'purchase_lines.lot_number',
                        'purchase_lines.exp_date',
                        'purchase_lines.quantity',
                        'purchase_lines.quantity_sold',
                        'purchase_lines.quantity_returned',
                        'purchase_lines.quantity_adjusted',
                        'business_locations.name as location_name'
                    )
                    ->orderBy('purchase_lines.exp_date', 'asc');

                $expiry_total = $expiry_query->count();
                $expiry_info = $expiry_query->skip(($expiry_page - 1) * $per_page)->take($per_page)->get()
                    ->map(function ($item) {
                        $remaining = $item->quantity - $item->quantity_sold - $item->quantity_returned - $item->quantity_adjusted;
                        $days_to_expiry = \Carbon::now()->diffInDays(\Carbon::parse($item->exp_date), false);
                        $status = 'ok';
                        if ($days_to_expiry < 0) {
                            $status = 'expired';
                        } elseif ($days_to_expiry <= 30) {
                            $status = 'expiring_soon';
                        }

                        return [
                            'lot_number' => $item->lot_number ?? 'N/A',
                            'exp_date' => $item->exp_date,
                            'location_name' => $item->location_name,
                            'stock' => $remaining,
                            'days_to_expiry' => $days_to_expiry,
                            'status' => $status
                        ];
                    });
            }

            return response()->json([
                'success' => true,
                'product' => [
                    'id' => $product->id,
                    'name' => $product->name,
                    'sku' => $product->sku,
                    'image_url' => $product->image_url,
                    'current_stock' => $current_stock,
                    'unit' => $product->unit->short_name ?? ''
                ],
                'sales' => [
                    'data' => $sales,
                    'total' => $sales_total,
                    'per_page' => $per_page,
                    'current_page' => (int)$sales_page,
                    'last_page' => ceil($sales_total / $per_page)
                ],
                'purchases' => [
                    'data' => $purchases,
                    'total' => $purchases_total,
                    'per_page' => $per_page,
                    'current_page' => (int)$purchases_page,
                    'last_page' => ceil($purchases_total / $per_page)
                ],
                'stock_history' => [
                    'data' => $stock_history,
                    'total' => $stock_total,
                    'per_page' => $per_page,
                    'current_page' => (int)$stock_page,
                    'last_page' => ceil($stock_total / $per_page)
                ],
                'warranty_sales' => [
                    'data' => $warranty_sales,
                    'total' => $warranty_total,
                    'per_page' => $per_page,
                    'current_page' => (int)$warranty_page,
                    'last_page' => ceil($warranty_total / $per_page)
                ],
                'expiry_info' => [
                    'data' => $expiry_info,
                    'total' => $expiry_total,
                    'per_page' => $per_page,
                    'current_page' => (int)$expiry_page,
                    'last_page' => ceil($expiry_total / $per_page)
                ],
                'permissions' => [
                    'can_view_sales' => $can_view_sales,
                    'can_view_purchases' => $can_view_purchases,
                    'can_view_stock' => $can_view_stock
                ]
            ]);

        } catch (\Exception $e) {
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            return response()->json([
                'success' => false,
                'msg' => __('messages.something_went_wrong')
            ]);
        }
    }
