// Product History Modal Functionality
var productHistoryCurrentSearch = '';
var productHistoryPages = {
    sales: 1,
    purchases: 1,
    stock: 1,
    warranty: 1,
    expiry: 1
};

$(document).ready(function() {
    // Focus on search input when modal opens
    $('#product_history_modal').on('shown.bs.modal', function() {
        $('#product_history_search').focus();
    });

    // Clear modal data when closed
    $('#product_history_modal').on('hidden.bs.modal', function() {
        $('#product_history_search').val('');
        $('#product_history_info').hide();
        $('#product_history_tabs_container').hide();
        productHistoryCurrentSearch = '';
        productHistoryPages = { sales: 1, purchases: 1, stock: 1, warranty: 1, expiry: 1 };
        resetProductHistoryTables();
    });

    // Search on button click
    $(document).on('click', '#search_product_history_btn', function() {
        productHistoryPages = { sales: 1, purchases: 1, stock: 1, warranty: 1, expiry: 1 };
        searchProductHistory();
    });

    // Search on Enter key
    $(document).on('keypress', '#product_history_search', function(e) {
        if (e.which === 13) {
            e.preventDefault();
            productHistoryPages = { sales: 1, purchases: 1, stock: 1, warranty: 1, expiry: 1 };
            searchProductHistory();
        }
    });

    // Pagination click handlers
    $(document).on('click', '.product-history-page', function(e) {
        e.preventDefault();
        var type = $(this).data('type');
        var page = $(this).data('page');
        productHistoryPages[type] = page;
        searchProductHistory(true);
    });
});

function searchProductHistory(isPagination) {
    var search = $('#product_history_search').val().trim();

    if (!search) {
        toastr.warning(LANG.please_enter_sku_or_product_name || 'Please enter SKU or product name');
        return;
    }

    productHistoryCurrentSearch = search;

    if (!isPagination) {
        $('#product_history_loading').show();
        $('#product_history_info').hide();
        $('#product_history_tabs_container').hide();
        resetProductHistoryTables();
    }

    $.ajax({
        url: '/sells/pos/get-product-history',
        method: 'GET',
        data: {
            search: search,
            sales_page: productHistoryPages.sales,
            purchases_page: productHistoryPages.purchases,
            stock_page: productHistoryPages.stock,
            warranty_page: productHistoryPages.warranty,
            expiry_page: productHistoryPages.expiry
        },
        dataType: 'json',
        success: function(response) {
            $('#product_history_loading').hide();

            if (response.success) {
                // Show product info
                $('#product_history_image').attr('src', response.product.image_url);
                $('#product_history_name').text(response.product.name);
                $('#product_history_sku').text(response.product.sku);
                $('#product_history_stock').text(
                    __number_f(response.product.current_stock, false, false, __currency_precision) +
                    ' ' + response.product.unit
                );
                $('#product_history_info').show();
                $('#product_history_tabs_container').show();

                // Populate Sales History
                populateSalesHistory(response.sales);

                // Populate Purchase History
                populatePurchaseHistory(response.purchases);

                // Populate Stock History
                populateStockHistory(response.stock_history);

                // Populate Warranty History
                populateWarrantyHistory(response.warranty_sales);

                // Populate Expiry Info
                populateExpiryHistory(response.expiry_info);

            } else {
                toastr.error(response.msg || 'Product not found');
            }
        },
        error: function() {
            $('#product_history_loading').hide();
            toastr.error(LANG.something_went_wrong || 'Something went wrong');
        }
    });
}

function resetProductHistoryTables() {
    var defaultMsg = '<tr><td colspan="7" class="text-center text-muted">' +
        (LANG.search_product_to_see_history || 'Search for a product to see history') + '</td></tr>';

    $('#sales_history_body').html(defaultMsg);
    $('#purchase_history_body').html(defaultMsg);
    $('#stock_history_body').html(defaultMsg);
    $('#warranty_history_body').html(defaultMsg);
    $('#expiry_history_body').html('<tr><td colspan="6" class="text-center text-muted">' +
        (LANG.search_product_to_see_history || 'Search for a product to see history') + '</td></tr>');

    $('#sales_pagination').html('');
    $('#purchases_pagination').html('');
    $('#stock_pagination').html('');
    $('#warranty_pagination').html('');
    $('#expiry_pagination').html('');
}

function generatePagination(type, pagination) {
    if (!pagination || pagination.last_page <= 1) {
        return '';
    }

    var html = '<ul class="pagination pagination-sm" style="margin: 10px 0;">';

    // Previous button
    if (pagination.current_page > 1) {
        html += '<li><a href="#" class="product-history-page" data-type="' + type + '" data-page="' + (pagination.current_page - 1) + '">&laquo;</a></li>';
    } else {
        html += '<li class="disabled"><span>&laquo;</span></li>';
    }

    // Page numbers
    var startPage = Math.max(1, pagination.current_page - 2);
    var endPage = Math.min(pagination.last_page, pagination.current_page + 2);

    if (startPage > 1) {
        html += '<li><a href="#" class="product-history-page" data-type="' + type + '" data-page="1">1</a></li>';
        if (startPage > 2) {
            html += '<li class="disabled"><span>...</span></li>';
        }
    }

    for (var i = startPage; i <= endPage; i++) {
        if (i === pagination.current_page) {
            html += '<li class="active"><span>' + i + '</span></li>';
        } else {
            html += '<li><a href="#" class="product-history-page" data-type="' + type + '" data-page="' + i + '">' + i + '</a></li>';
        }
    }

    if (endPage < pagination.last_page) {
        if (endPage < pagination.last_page - 1) {
            html += '<li class="disabled"><span>...</span></li>';
        }
        html += '<li><a href="#" class="product-history-page" data-type="' + type + '" data-page="' + pagination.last_page + '">' + pagination.last_page + '</a></li>';
    }

    // Next button
    if (pagination.current_page < pagination.last_page) {
        html += '<li><a href="#" class="product-history-page" data-type="' + type + '" data-page="' + (pagination.current_page + 1) + '">&raquo;</a></li>';
    } else {
        html += '<li class="disabled"><span>&raquo;</span></li>';
    }

    html += '</ul>';
    html += '<small class="text-muted">' + (LANG.showing || 'Showing') + ' ' +
        ((pagination.current_page - 1) * pagination.per_page + 1) + '-' +
        Math.min(pagination.current_page * pagination.per_page, pagination.total) +
        ' ' + (LANG.of || 'of') + ' ' + pagination.total + '</small>';

    return html;
}

function populateSalesHistory(salesData) {
    var html = '';
    var sales = salesData.data || [];

    if (sales && sales.length > 0) {
        $.each(sales, function(index, sale) {
            var subtotal = parseFloat(sale.unit_price_inc_tax) * parseFloat(sale.quantity);
            html += '<tr>' +
                '<td>' + sale.invoice_no + '</td>' +
                '<td>' + formatDate(sale.transaction_date) + '</td>' +
                '<td>' + sale.customer_name + '</td>' +
                '<td>' + __number_f(sale.quantity, false, false, __currency_precision) + '</td>' +
                '<td>' + __currency_trans_from_en(sale.unit_price_inc_tax, true) + '</td>' +
                '<td>' + __currency_trans_from_en(subtotal, true) + '</td>' +
                '<td>' + sale.location_name + '</td>' +
                '</tr>';
        });
    } else {
        html = '<tr><td colspan="7" class="text-center text-muted">' +
            (LANG.no_records_found || 'No records found') + '</td></tr>';
    }
    $('#sales_history_body').html(html);
    $('#sales_pagination').html(generatePagination('sales', salesData));
}

function populatePurchaseHistory(purchasesData) {
    var html = '';
    var purchases = purchasesData.data || [];

    if (purchases && purchases.length > 0) {
        $.each(purchases, function(index, purchase) {
            var subtotal = parseFloat(purchase.purchase_price_inc_tax) * parseFloat(purchase.quantity);
            html += '<tr>' +
                '<td>' + purchase.ref_no + '</td>' +
                '<td>' + formatDate(purchase.transaction_date) + '</td>' +
                '<td>' + purchase.supplier_name + '</td>' +
                '<td>' + __number_f(purchase.quantity, false, false, __currency_precision) + '</td>' +
                '<td>' + __currency_trans_from_en(purchase.purchase_price_inc_tax, true) + '</td>' +
                '<td>' + __currency_trans_from_en(subtotal, true) + '</td>' +
                '<td>' + purchase.location_name + '</td>' +
                '</tr>';
        });
    } else {
        html = '<tr><td colspan="7" class="text-center text-muted">' +
            (LANG.no_records_found || 'No records found') + '</td></tr>';
    }
    $('#purchase_history_body').html(html);
    $('#purchases_pagination').html(generatePagination('purchases', purchasesData));
}

function populateStockHistory(stockData) {
    var html = '';
    var stockHistory = stockData.data || [];

    if (stockHistory && stockHistory.length > 0) {
        $.each(stockHistory, function(index, item) {
            html += '<tr>' +
                '<td>' + (item.transaction_date || '-') + '</td>' +
                '<td>' + (item.type || '-') + '</td>' +
                '<td>' + (item.ref_no || '-') + '</td>' +
                '<td>' + (item.location || '-') + '</td>' +
                '<td class="text-success">' + (item.quantity_in || '0') + '</td>' +
                '<td class="text-danger">' + (item.quantity_out || '0') + '</td>' +
                '<td>' + (item.stock || '0') + '</td>' +
                '</tr>';
        });
    } else {
        html = '<tr><td colspan="7" class="text-center text-muted">' +
            (LANG.no_records_found || 'No records found') + '</td></tr>';
    }
    $('#stock_history_body').html(html);
    $('#stock_pagination').html(generatePagination('stock', stockData));
}

function populateWarrantyHistory(warrantyData) {
    var html = '';
    var warrantySales = warrantyData.data || [];

    if (warrantySales && warrantySales.length > 0) {
        $.each(warrantySales, function(index, item) {
            var statusClass = item.is_active ? 'label-success' : 'label-danger';
            var statusText = item.is_active ?
                (LANG.active || 'Active') : (LANG.expired || 'Expired');

            html += '<tr>' +
                '<td>' + item.invoice_no + '</td>' +
                '<td>' + formatDate(item.transaction_date) + '</td>' +
                '<td>' + item.customer_name + '</td>' +
                '<td>' + item.warranty_name + '</td>' +
                '<td>' + formatDate(item.warranty_start) + '</td>' +
                '<td>' + formatDate(item.warranty_end) + '</td>' +
                '<td><span class="label ' + statusClass + '">' + statusText + '</span></td>' +
                '</tr>';
        });
    } else {
        html = '<tr><td colspan="7" class="text-center text-muted">' +
            (LANG.no_warranty_info || 'No warranty information available') + '</td></tr>';
    }
    $('#warranty_history_body').html(html);
    $('#warranty_pagination').html(generatePagination('warranty', warrantyData));
}

function populateExpiryHistory(expiryData) {
    var html = '';
    var expiryInfo = expiryData.data || [];

    if (expiryInfo && expiryInfo.length > 0) {
        $.each(expiryInfo, function(index, item) {
            var statusClass = 'label-success';
            var statusText = LANG.ok || 'OK';

            if (item.status === 'expired') {
                statusClass = 'label-danger';
                statusText = LANG.expired || 'Expired';
            } else if (item.status === 'expiring_soon') {
                statusClass = 'label-warning';
                statusText = LANG.expiring_soon || 'Expiring Soon';
            }

            html += '<tr>' +
                '<td>' + item.lot_number + '</td>' +
                '<td>' + formatDate(item.exp_date) + '</td>' +
                '<td>' + item.location_name + '</td>' +
                '<td>' + __number_f(item.stock, false, false, __currency_precision) + '</td>' +
                '<td>' + item.days_to_expiry + ' ' + (LANG.days || 'days') + '</td>' +
                '<td><span class="label ' + statusClass + '">' + statusText + '</span></td>' +
                '</tr>';
        });
    } else {
        html = '<tr><td colspan="6" class="text-center text-muted">' +
            (LANG.no_expiry_info || 'No expiry information available') + '</td></tr>';
    }
    $('#expiry_history_body').html(html);
    $('#expiry_pagination').html(generatePagination('expiry', expiryData));
}

function formatDate(dateString) {
    if (!dateString) return '-';
    try {
        var date = new Date(dateString);
        return date.toLocaleDateString() + ' ' + date.toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'});
    } catch (e) {
        return dateString;
    }
}
